/*
 * Decompiled with CFR 0.152.
 */
package com.google.photos.library.v1.upload;

import com.google.api.gax.retrying.RetrySettings;
import com.google.api.gax.rpc.ClientContext;
import com.google.photos.library.v1.PhotosLibrarySettings;
import com.google.photos.library.v1.upload.UploadMediaItemRequest;
import com.google.photos.library.v1.upload.UploadMediaItemResponse;
import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.Callable;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.EntityBuilder;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.threeten.bp.Duration;
import org.threeten.bp.temporal.ChronoUnit;
import org.threeten.bp.temporal.TemporalUnit;

final class PhotosLibraryUploadCallable
implements Callable<UploadMediaItemResponse> {
    private static final String FILE_NAME_HEADER = "X-Goog-Upload-File-Name";
    private static final String FILE_SIZE_HEADER = "X-Goog-Upload-Raw-Size";
    private static final String UPLOAD_PROTOCOL_HEADER = "X-Goog-Upload-Protocol";
    private static final String UPLOAD_COMMAND_HEADER = "X-Goog-Upload-Command";
    private static final String UPLOAD_CONTENT_TYPE_HEADER = "X-Goog-Upload-Content-Type";
    private static final String UPLOAD_STATUS_HEADER = "X-Goog-Upload-Status";
    private static final String UPLOAD_URL_HEADER = "X-Goog-Upload-URL";
    private static final String UPLOAD_OFFSET_HEADER = "X-Goog-Upload-Offset";
    private static final String UPLOAD_SIZE_HEADER = "X-Goog-Upload-Header-Content-Length";
    private static final String UPLOAD_GRANULARITY_HEADER = "X-Goog-Upload-Chunk-Granularity";
    private static final String RECEIVED_BYTE_COUNT_HEADER = "X-Goog-Upload-Size-Received";
    private static final String UPLOAD_PROTOCOL_VALUE = "resumable";
    private static final Duration UNLIMITED_TIMEOUT = Duration.ZERO;
    private static final int UNLIMITED_RETRIES = 0;
    private final UploadMediaItemRequest request;
    private final ClientContext clientContext;
    private final PhotosLibrarySettings photosLibrarySettings;
    private final AtomicReference<String> atomicResumeUrl;
    private int optimalChunkSize;

    PhotosLibraryUploadCallable(UploadMediaItemRequest request, ClientContext clientContext, PhotosLibrarySettings photosLibrarySettings) {
        this.request = request;
        this.clientContext = clientContext;
        this.photosLibrarySettings = photosLibrarySettings;
        this.atomicResumeUrl = new AtomicReference();
    }

    public AtomicReference<String> getAtomicResumeUrl() {
        return this.atomicResumeUrl;
    }

    @Override
    public UploadMediaItemResponse call() throws Exception {
        long initialMillis = this.clientContext.getClock().millisTime();
        String uploadUrl = this.getUploadUrl();
        this.atomicResumeUrl.set(uploadUrl);
        this.checkForTimeout(initialMillis);
        Optional<HttpResponse> response = Optional.empty();
        RetrySettings retrySettings = this.photosLibrarySettings.uploadMediaItemSettings().getRetrySettings();
        boolean successful = false;
        int retries = 0;
        OptionalLong previousDelayMillis = OptionalLong.empty();
        while (!(successful || retrySettings.getMaxAttempts() != 0 && retries >= retrySettings.getMaxAttempts())) {
            ++retries;
            long receivedByteCount = this.getReceivedByteCount(uploadUrl);
            this.request.seekCursor(receivedByteCount);
            successful = true;
            while (receivedByteCount < this.request.getFileSize()) {
                response = Optional.of(this.uploadNextChunk(uploadUrl, receivedByteCount));
                this.checkForTimeout(initialMillis);
                if (!PhotosLibraryUploadCallable.isStatusOk(response.get().getStatusLine().getStatusCode())) {
                    successful = false;
                    break;
                }
                receivedByteCount = this.getReceivedByteCount(uploadUrl);
                this.checkForTimeout(initialMillis);
            }
            if (successful || retries >= retrySettings.getMaxAttempts()) continue;
            long delayMillis = retrySettings.getInitialRetryDelay().get((TemporalUnit)ChronoUnit.MILLIS);
            if (previousDelayMillis.isPresent()) {
                delayMillis = (long)((double)previousDelayMillis.getAsLong() * retrySettings.getRetryDelayMultiplier());
            }
            long actualDelayMillis = Math.min(delayMillis, retrySettings.getMaxRetryDelay().get((TemporalUnit)ChronoUnit.MILLIS));
            if (retrySettings.isJittered()) {
                actualDelayMillis = ThreadLocalRandom.current().nextLong(actualDelayMillis);
            }
            Thread.sleep(actualDelayMillis);
            this.checkForTimeout(initialMillis);
            previousDelayMillis = OptionalLong.of(actualDelayMillis);
        }
        if (!successful) {
            if (response.isPresent()) {
                throw new HttpResponseException(response.get().getStatusLine().getStatusCode(), "The upload was completed but failed to finalize or get the result.");
            }
            throw new IllegalStateException("The upload encountered an unknown error.");
        }
        return this.buildUploadMediaItemResponse(response);
    }

    private void checkForTimeout(long initialMillis) throws TimeoutException {
        long duration;
        if (this.photosLibrarySettings.uploadMediaItemSettings().getRetrySettings().getTotalTimeout() != UNLIMITED_TIMEOUT && (duration = this.clientContext.getClock().millisTime() - initialMillis) > this.photosLibrarySettings.uploadMediaItemSettings().getRetrySettings().getTotalTimeout().get((TemporalUnit)ChronoUnit.MILLIS)) {
            throw new TimeoutException("The upload has timed out.");
        }
    }

    private String getUploadUrl() throws IOException {
        CloseableHttpClient httpClient;
        CloseableHttpResponse response;
        if (this.request.getUploadUrl().isPresent()) {
            return this.request.getUploadUrl().get();
        }
        HttpPost httpPost = this.createAuthenticatedPostRequest(PhotosLibrarySettings.getUploadEndpoint());
        httpPost.addHeader(UPLOAD_PROTOCOL_HEADER, UPLOAD_PROTOCOL_VALUE);
        httpPost.addHeader(UPLOAD_COMMAND_HEADER, "start");
        httpPost.addHeader(FILE_SIZE_HEADER, String.valueOf(this.request.getFileSize()));
        if (this.request.getMimeType().isPresent()) {
            httpPost.addHeader(UPLOAD_CONTENT_TYPE_HEADER, this.request.getMimeType().get());
        }
        if (this.request.getFileName().isPresent()) {
            httpPost.addHeader(FILE_NAME_HEADER, this.request.getFileName().get());
        }
        if ((response = (httpClient = HttpClientBuilder.create().useSystemProperties().build()).execute((HttpUriRequest)httpPost)).getFirstHeader(UPLOAD_GRANULARITY_HEADER) != null) {
            this.updateOptimalChunkSize(Integer.parseInt(response.getFirstHeader(UPLOAD_GRANULARITY_HEADER).getValue()));
        }
        switch (response.getFirstHeader(UPLOAD_STATUS_HEADER).getValue()) {
            case "active": {
                return response.getFirstHeader(UPLOAD_URL_HEADER).getValue();
            }
            case "final": {
                throw new IllegalArgumentException("The upload url is either finalized or rejected by the server.");
            }
        }
        throw new IllegalStateException("Invalid upload status received.");
    }

    private long getReceivedByteCount(String uploadUrl) throws IOException {
        HttpPost httpPost = this.createAuthenticatedPostRequest(uploadUrl);
        httpPost.addHeader(UPLOAD_PROTOCOL_HEADER, UPLOAD_PROTOCOL_VALUE);
        httpPost.addHeader(UPLOAD_COMMAND_HEADER, "query");
        CloseableHttpClient httpClient = HttpClientBuilder.create().useSystemProperties().build();
        CloseableHttpResponse response = httpClient.execute((HttpUriRequest)httpPost);
        if (response.getFirstHeader(UPLOAD_GRANULARITY_HEADER) != null) {
            this.updateOptimalChunkSize(Integer.parseInt(response.getFirstHeader(UPLOAD_GRANULARITY_HEADER).getValue()));
        }
        switch (response.getFirstHeader(UPLOAD_STATUS_HEADER).getValue()) {
            case "active": {
                return Long.parseLong(response.getFirstHeader(RECEIVED_BYTE_COUNT_HEADER).getValue());
            }
            case "final": {
                return this.request.getFileSize();
            }
        }
        throw new IllegalStateException("Invalid upload status received.");
    }

    private HttpResponse uploadNextChunk(String uploadUrl, long receivedByteCount) throws IOException {
        byte[] dataChunk = new byte[this.optimalChunkSize];
        int readByteCount = this.request.readData(dataChunk);
        if (readByteCount < this.optimalChunkSize) {
            dataChunk = PhotosLibraryUploadCallable.trimByteArray(dataChunk, readByteCount);
        }
        HttpPost httpPost = this.createAuthenticatedPostRequest(uploadUrl);
        httpPost.addHeader(UPLOAD_PROTOCOL_HEADER, UPLOAD_PROTOCOL_VALUE);
        if (receivedByteCount + (long)readByteCount == this.request.getFileSize()) {
            httpPost.addHeader(UPLOAD_COMMAND_HEADER, String.join((CharSequence)",", "upload", "finalize"));
        } else {
            httpPost.addHeader(UPLOAD_COMMAND_HEADER, "upload");
        }
        httpPost.addHeader(UPLOAD_OFFSET_HEADER, String.valueOf(receivedByteCount));
        httpPost.addHeader(UPLOAD_SIZE_HEADER, String.valueOf(readByteCount));
        httpPost.setEntity(EntityBuilder.create().setBinary(dataChunk).build());
        CloseableHttpClient httpClient = HttpClientBuilder.create().useSystemProperties().setDefaultRequestConfig(this.getRequestConfig()).build();
        return httpClient.execute((HttpUriRequest)httpPost);
    }

    private RequestConfig getRequestConfig() {
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        if (this.photosLibrarySettings.uploadMediaItemSettings().getRetrySettings().getTotalTimeout() != UNLIMITED_TIMEOUT) {
            configBuilder.setConnectionRequestTimeout(Math.toIntExact(this.photosLibrarySettings.uploadMediaItemSettings().getRetrySettings().getTotalTimeout().get((TemporalUnit)ChronoUnit.MILLIS)));
        }
        return configBuilder.build();
    }

    private UploadMediaItemResponse buildUploadMediaItemResponse(Optional<HttpResponse> response) throws IOException {
        if (!response.isPresent() || !PhotosLibraryUploadCallable.isStatusOk(response.get().getStatusLine().getStatusCode())) {
            throw new IllegalArgumentException("The upload was completed but failed to finalize or get the result.");
        }
        return UploadMediaItemResponse.newBuilder().setUploadToken(EntityUtils.toString((HttpEntity)response.get().getEntity())).build();
    }

    private HttpPost createAuthenticatedPostRequest(String uploadUrl) throws IOException {
        HttpPost request = new HttpPost(uploadUrl);
        Map requestMetadata = this.clientContext.getCredentials().getRequestMetadata();
        for (Map.Entry entry : requestMetadata.entrySet()) {
            request.addHeader((String)entry.getKey(), String.join((CharSequence)", ", (Iterable)entry.getValue()));
        }
        return request;
    }

    private void updateOptimalChunkSize(int uploadGranularity) {
        this.optimalChunkSize = (1 + (this.request.getChunkSize() - 1) / uploadGranularity) * uploadGranularity;
    }

    private static boolean isStatusOk(int statusCode) {
        return statusCode == 200;
    }

    private static byte[] trimByteArray(byte[] originalBytes, int newSize) {
        if (newSize > originalBytes.length) {
            return originalBytes;
        }
        byte[] newBytes = new byte[newSize];
        System.arraycopy(originalBytes, 0, newBytes, 0, newSize);
        return newBytes;
    }

    private static final class UploadStatuses {
        private static final String ACTIVE = "active";
        private static final String FINAL = "final";

        private UploadStatuses() {
        }
    }

    private static final class UploadCommands {
        private static final String START = "start";
        private static final String UPLOAD = "upload";
        private static final String QUERY = "query";
        private static final String FINALIZE = "finalize";
        private static final String CANCEL = "cancel";

        private UploadCommands() {
        }
    }
}

